using System;
using System.IO;

namespace Team_Project.PersistencyManagers.Storages
{
	/// <summary>
	/// Ogni storage deve implementare i metodi di questa interfaccia per
	/// permettere la persistenza di elementi in team-project
	/// </summary>
	public interface IStorage
	{
		/// <summary>
		/// Apre uno stream in lettura sull'elemento specificato
		/// </summary>
		/// <param name="Location">La "location" dove si trova l'elemento</param>
		/// <param name="Name">Il nome completo (compresa l'eventuale estensione)
		/// dell'elemento da aprire</param>
		/// <returns>Uno stream in lettura con i dati dell'elemento</returns>
		Stream OpenRead(string Location, string Name);
		/// <summary>
		/// Apre uno stream in scrittura sull'elemento specificato
		/// </summary>
		/// <param name="Location">La "location" dove scrivere l'elemtno</param>
		/// <param name="Name">Il nome completo (compresa l'eventuale estensione)
		/// dell'elemento da aprire</param>
		/// <returns>Uno stream in scrittura dove inserire i dati dell'elemento</returns>
		Stream OpenWrite(string Location, string Name);


		/// <summary>
		/// Controlla l'esistenza di una location
		/// </summary>
		/// <param name="Location">Percorso della location da controllare</param>
		/// <returns><see langword="false"/> se la location esiste, <see langword="false"/> altrimenti</returns>
		bool LocationExists(string Location);
		/// <summary>
		/// Controlla l'esistenza di un elemento
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="Name"></param>
		/// <returns><see langword="true"/> se l'elemento esiste, <see langword="false"/> altrimenti</returns>
		bool Exists(string Location, string Name);

		/// <summary>
		/// Funzione utilizzata per l'esplorazione delle location.
		/// Permette di esplorare le location innestate in altre locations o 
		/// presenti nella radice
		/// </summary>
		/// <param name="baseL">Percorso della location di cui ottenere le 
		/// "sotto locations". Utilizzare la stringa vuota se si desidera ottenere
		/// le location di primo livello</param>
		/// <returns>Un array di stringhe contenente i nomi di tutte le sotto 
		/// locations trovate. Se non ci sono sotto location l'array  lungo 0</returns>
		string[] LocationsList(string baseL);

		/// <summary>
		/// Funzione utilizzata per ottenere gli elementi contenuti in una location.
		/// </summary>
		/// <param name="Location">Percorso della location di cui ottenere il contenuto</param>
		/// <returns>Un array di stringhe contenente il nome completo degli elementi
		/// contenuti nella location.</returns>
		string[] ElementsIn(string Location);

		/// <summary>
		/// Crea una nuova location con il nome specificato
		/// </summary>
		/// <param name="Location">Percorso completo della location da creare</param>
		void CreateLocation(string Location);
		/// <summary>
		/// Distrugge una location esistente. A seconda del parametro <paramref name="checkEmpty"/>
		/// la funzione pu cancellare ricorsivamente il contenuto della location o lanciare un'eccezione
		/// nel caso essa non sia vuota
		/// </summary>
		/// <param name="Location">Percorso della location da eliminare</param>
		/// <param name="checkEmpty">Se <see langword="true"/> la funzione lancia
		/// un'eccezione nel caso la location no sia vuota, altrimenti cancella
		/// ricorsivamente il suo contenuto</param>
		void DestroyLocation(string Location, bool checkEmpty);

		/// <summary>
		/// Elimina un elemento esistente
		/// </summary>
		/// <param name="Location">Percorso della location che contiene l'elemento</param>
		/// <param name="Name">Nome completo dell'elemento da eliminare</param>
		void Delete(string Location, string Name);
	}
}